	%==========================================================================
	% This is a general-purpose MATLAB script for designing classical recursive  
	% digital filters. This version was prepared by Tapio on October 13, 2018. 
	% For more detail, go through this script and run the program. Read also 
	% Tapio Saramki, "Simple Formulae for Designing Classical Recursive 
	% Digital Filters"
	%==========================================================================
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 
	
	%	Author: 			Tapio Saramaki 2018-11-13
	%	Modified by:	
	%	Copyright:			by authors - not released for commercial use
	%	Version:			1	 
	%	Known bugs:			None
	%	Report bugs to:		tapio.saramaki@tut.fi
	
	close all;
	disp(' ')
	disp(['HI OVER THERE! I am a program for synthesizing classical ',...
	'recursive digital filters.'])
	disp(' ')
	disp(['FIRST, I should know how to design the very classical lowpass, ',...
	'highpass, bandpass, and bandstop filters.'])
	itype=input(['Hence, type "1" for lowpass, "2" highpass, "3" ',...
	'bandpass, or "4" for bandstop: ']);
	disp(' ')
	if itype==1 | itype==2
	    if itype==1
	        disp('You are interested in the LOWPASS FILTER.')
		else 
	        disp('You are interested in the HIGHPASS FILTER.')
		end
	    disp(['In this case, the design starts with specifying the '...
		'following four sets of parameters:'])
	    disp(['(I)   omega_p, the passband edge angle given as a fraction ',...
		'of pi or half the sampling rate'])
	    disp(['(II)  omega_s, the stopband edge angle given as a fraction ',...
		'of pi or half the sampling rate'])
	    if itype==1
	        disp('such that omega_p < omega_s')
		else
	        disp('such that omega_s < omega_p')
		end
	end
	if itype==3 | itype==4
	    if itype==3
	        disp('You are interested in the BANDPASS FILTER.')
		else 
	        disp('You are interested in the BANDSTOP FILTER.')
		end
	    disp(['In this case, the design starts with specifying the ',...
		'following four parameters:'])
	    disp(['(I)   omega_p1 and omega_p2, the passband edge angles ',...
		'given as a fraction of pi or half the sampling rate'])
	    disp(['(II)  omega_s1 and omega_s2, the stopband edge angles given',...
		' as a fraction of pi or half the sampling rate'])
	    if itype==3
	        disp('such that omega_s1 < omega_p1 < omega_p2 < omega_s2') 
		else
	        disp('such that omega_p1 < omega_s1 < omega_s2 < omega_p2')
		end
	end
	disp('(III) A_p, the passband variation in dB (positive)')
	disp('(IV)  A_s, the minimum stopband attenuation in dB (positive)')
	disp('HENCE, SECONDLY, specify')
	if itype==1 
	    omega_p=input('omega_p = ');
	    omega_s=input('omega_s = ');
	elseif itype==2
	    theta_s=input('omega_s = ');
	    theta_p=input('omega_p = ');
	elseif itype==3
		omega_s1=input('omega_s1 = ');
		omega_p1=input('omega_p1 = '); 
		omega_p2=input('omega_p2 = '); 
		omega_s2=input('omega_s2 = ');
	else
		omega_p1=input('omega_p1 = ');
		omega_s1=input('omega_s1 = '); 
		omega_s2=input('omega_s2 = '); 
		omega_p2=input('omega_p2 = ');
	end
	A_p=input('A_p = ');
	A_s=input('A_s = ');
	disp(' ')
	disp(['THIRD,  give "1" if the resulting filter is desired to be ',...
	'implementable as a parallel'])
	Jopt=input(['connection of two (real-valued) all-pass filters. ',...
	'Otherwise, whatever: ']);
	disp(' ')
	disp(' ')
	disp(['OKEY DOKEY - after some brainstorming, I ended up with the ',...
	'following:'])
	if itype==2
	    %======================================================================
	    % Lowpass-to-highpass transformation in use is the simplest one:
	    %  z^[-1}-->-z^[-1}. The angular frequency of the prototype filter, 
	    % theta,  and that of the desired filter, omega, are related trough
	    % theta = pi - omega.
	    %======================================================================
	    % Passband and stopband edges of the lowpass prototype
	    %======================================================================
	    omega_p=1-theta_p;
	    omega_s=1-theta_s;
	end    
	if itype==3
	    %======================================================================
	    % Lowpass-to-bandpass transformation in use: determine the parameters
	    % of the transformation as well as the passband and stopband edges of 
	    % the lowpass prototype filter.
	    % Here the ideas given on pages 72 and 73 in 
	    % http://www.cs.tut.fi/~ts/part4.pdf are followed in order to make the 
	    % synthesis of bandpass filters extremely simple.
	    %======================================================================
	    % Fixed passband edge of the lowpass prototype filter
	    %----------------------------------------------------------------------
	    omega_p=.5;
	    %----------------------------------------------------------------------
	    % Specify the lowpass-to-bandpass transformation of the form
	    %
	    %            gamma_2-gamma_1z^{-1}+z^{-2}
	    % Z^{-1} = - ------------------------------
	    %            1-gamma_1z^{-1}+gamma_2z^{-2}
	    %
	    % Note that the passband edge of the prototype, for the sake of 
	    % simplicity, is fixed to be theta_p=pi/2.
	    %----------------------------------------------------------------------
	    alpha=...
		cos((omega_p2*pi+omega_p1*pi)/2)/cos((omega_p2*pi-omega_p1*pi)/2);
	    K=cot((omega_p2*pi-omega_p1*pi)/2);
	    gamma_1=2*alpha*K/(K+1);
	    gamma_2=(K-1)/(K+1);
	    %----------------------------------------------------------------------
	    % Determine the required stopband edge of the lowpass prototype filter
	    %----------------------------------------------------------------------
	    Ome_s=[omega_s1 omega_s2];
	    y=gamma_1*sin(pi*Ome_s)-gamma_2*sin(2*pi*Ome_s);
	    x=1-gamma_1*cos(pi*Ome_s)+gamma_2*cos(2*pi*Ome_s);
	    Thet_s=-pi+2*Ome_s*pi+2*atan2(y,x);
	    omega_s=min(abs(Thet_s(1)),abs(Thet_s(2)))/pi;
	end
	if itype==4
	    %======================================================================
	    % Lowpass-to-bandstop transformation in use: determine the parameters
	    % of the transformation as well as the passband and stopband edges of 
	    % the low-pass prototype filter
	    % Here the ideas given on pages 74 and 75 in 
	    % http://www.cs.tut.fi/~ts/part4.pdf are followed in order to make the 
	    % synthesis of bandpass filters extremely simple.
	    %======================================================================
	    % Fixed passband edge of the low-pass prototype filter
	    %----------------------------------------------------------------------
	    omega_p=.5;
	    %----------------------------------------------------------------------
	    % Specify the lowpass-to-bandstop transformation of the form
	    %
	    %           gamma_2-gamma_1z^{-1}+z^{-2}
	    % Z^{-1} =  ----------------------------
	    %           1-gamma_1z^{-1}+gamma_2z^{-2}
	    %
	    % Similarly, as for the lowpass-to-bandpass tranformation, the passband 
	    % edge of the prototype is fixed to be theta_p=pi/2.
	    %----------------------------------------------------------------------
	    alpha=...
		cos((omega_p2*pi+omega_p1*pi)/2)/cos((omega_p2*pi-omega_p1*pi)/2);
	    K=tan((omega_p2*pi-omega_p1*pi)/2);
	    gamma_1=2*alpha/(K+1);
	    gamma_2=(1-K)/(1+K);
	    %----------------------------------------------------------------------
	    % Determine the required stopband edge of the lowpass prototype filter
	    %----------------------------------------------------------------------
	    Ome_s=[omega_s1 omega_s2];
	    y=gamma_1*sin(pi*Ome_s)-gamma_2*sin(2*pi*Ome_s);
	    x=1-gamma_1*cos(pi*Ome_s)+gamma_2*cos(2*pi*Ome_s);
	    Thet_s=2*Ome_s*pi+2*atan2(y,x);
	    omega_s=min(Thet_s(1),2*pi-Thet_s(2))/pi;
	end
	%==========================================================================
	% Minimum orders for the lowpass prototype filter, which is a Butterworth, 
	% Chebyshev, inverse Chebyshev, or Cauer (elliptic) filter.
	% Maximized values of omega_p and A_s as well as minimized values of
	% omega_s and A_p are simultaneously determined for each of the four filter
	% types.
	%==========================================================================
	[NN_low,new_omega_p,new_omega_s,new_A_p,new_A_s]=...
		ClassicIIR_LP_allpara(omega_p, omega_s, A_p, A_s);
	%==========================================================================
	% Orders of the overall filter
	%==========================================================================
	if itype==3 | itype==4
	    multiply=2;
	else
	    multiply=1;
	end
	NN=multiply*NN_low;
	disp('FIRST,')
	disp('The minimum orders to meet the criteria are');
	fprintf('Butterworth filter: order = %d\n',NN(1));
	fprintf('Chebyshev filter:   order = %d\n',NN(2));
	fprintf('Inverse Chebyshev:  order = %d\n',NN(3));
	fprintf('Cauer filter:       order = %d\n',NN(4));
	%==========================================================================
	% Check whether the overall filter is desired to be implemented as a 
	% parallel connection of two real-valued all-pass filters. If so, change 
	% the orders accordingly. What is needed in this case is that the order of 
	% the prototype should be odd. For this purpose, use 
	% ClassicIIR_LP_allpara_LWD, instead of ClassicIIR_LP_allpara. 
	%==========================================================================
	if Jopt==1
	    [NN_low,new_omega_p,new_omega_s,new_A_p,new_A_s]=...
		ClassicIIR_LP_allpara_LWD(omega_p, omega_s, A_p, A_s);
	    
	    % Orders of the overall filter
	    
	    NN=multiply*NN_low;
	    disp(' ')
	    disp('You were keen on implementing the filter AS A ')
	    disp('PARALLEL CONNECTION OF TWO ALL-PASS FILTERS')
	    disp(' ')
	    %disp('You were keen on a LWD implementation.') 
	    disp('In this case, the corresponding minimum orders are');
	    fprintf('Butterworth filter: order = %d\n',NN(1));
	    fprintf('Chebyshev filter:   order = %d\n',NN(2));
	    fprintf('Inverse Chebyshev:  order = %d\n',NN(3));
	    fprintf('Cauer filter:       order = %d\n',NN(4));
	end
	%==========================================================================
	% Selection within Butterworth, Chebyshev, inverse Chebyshev, and Cauer 
	% filters
	%==========================================================================
	disp(' ')
	disp('It is your choice: Type "1" for Butterworth, "2" for Chebyshev,')
	iirtype=input(['                        "3" for inverse Chebyshev, ',...
	'and "4" for Cauer: ']);
	%==========================================================================
	% Four options to allocate the additional design margin caused by selecting
	% the filter order to be an integer or an odd integer that is larger than 
	% or equal to the decimal order, by which the given criteria are exactly 
	% met.
	%==========================================================================
	N_low=NN_low(iirtype);
	oomega_p=new_omega_p(iirtype);
	oomega_s=new_omega_s(iirtype);
	AA_p=new_A_p(iirtype);
	AA_s=new_A_s(iirtype);
	%==========================================================================
	% "Real" and "maximized" passband edges for bandpass and bandstop filters 
	% and the corresponding "real" and "minimized" stopband edges 
	%==========================================================================
	if itype==3 | itype==4
	    %======================================================================
	    % "Real" stopband egdes
	    %======================================================================
	    zero_LP = exp(j*omega_s*pi);
	    %c(1) = 1+gamma_2*zero_LP;
	    if itype==3
	        c(1) = gamma_2*zero_LP+1;
	        c(2) = -gamma_1*(1+zero_LP);
	        c(3) = zero_LP+gamma_2;
		else
	        c(1) = gamma_2*zero_LP-1;
	        c(2) = -gamma_1*(zero_LP-1);
	        c(3) = zero_LP-gamma_2;
		end
	    zero_BPBS = roots(c);
	    real_omegas=sort(abs(angle(zero_BPBS)))/pi;
	    %======================================================================
	    % "Minimized" stopband egdes
	    %======================================================================
	    zero_LP = exp(j*oomega_s*pi);
	    if itype==3
	        c(1) = gamma_2*zero_LP+1;
	        c(2) = -gamma_1*(1+zero_LP);
	        c(3) = zero_LP+gamma_2;
		else
	        c(1) = gamma_2*zero_LP-1;
	        c(2) = -gamma_1*(zero_LP-1);
	        c(3) = zero_LP-gamma_2;
		end
	    zero_BPBS = roots(c);
	    min_omegas=sort(abs(angle(zero_BPBS)))/pi;
	    %======================================================================
	    % "Real" passband egdes
	    %======================================================================
	    real_omegap=[omega_p1 omega_p2];    
	    %======================================================================
	    % "Maximized" passband edges 
	    %======================================================================
	    zero_LP = exp(j*oomega_p*pi);
	    if itype==3
	        c(1) = gamma_2*zero_LP+1;
	        c(2) = -gamma_1*(1+zero_LP);
	        c(3) = zero_LP+gamma_2;
		else
	        c(1) = gamma_2*zero_LP-1;
	        c(2) = -gamma_1*(zero_LP-1);
	        c(3) = zero_LP-gamma_2;
		end
	    zero_BPBS = roots(c);
	    max_omegap=sort(abs(angle(zero_BPBS)))/pi;
	end
	%==========================================================================
	% Determine the overall filter order depending on whether a lowpass, 
	% highpass, bandpass, or bandstop filter is under consideration
	%==========================================================================
	if itype==3 | itype==4
	    N=2*N_low;
	else
	    N=N_low;
	end
	fprintf(['SECONDLY, for the filter order N = %d, there are the ',...
	'following four options:\n'], N')
	if itype==1
		fprintf(['(A) Maximized omega_p = %.5f*pi, instead of the specified ',...
	    'omega_p = %.5f*pi \n'], oomega_p,omega_p')
	elseif itype==2
		fprintf(['(A) Minimized omega_p = %.5f*pi, instead of the specified ',...
	    'omega_p = %.5f*pi \n'],1-oomega_p,1-omega_p')   
	else
		fprintf(['(A) Transition bandwidths are minimized by omega_p1 = ',...
	    '%.5f*pi and omega_p2 = %.5f*pi,\n'],max_omegap(1), max_omegap(2)')
		fprintf(['    instead of the specified omega_p1 = %.5f*pi ',...
	    'and omega_p2 = %.5f*pi \n'], omega_p1, omega_p2')
	end
	if itype==1 
		fprintf(['(B) Minimzed omega_s = %.5f*pi, ',...
	    'instead of specified omega_s = %.5f*pi  \n'], oomega_s, omega_s')
	elseif itype==2
		fprintf(['(B) Maximized  omega_s = %.5f*pi, ',...
	    'instead of specified omega_s = %.5f*pi  \n'], 1-oomega_s, 1-omega_s')    
	else
		fprintf(['(B) Transition bandwidths are minimized by omega_s1 = ',...
	    '%.5f*pi and omega_s2 = ',...
	    '%.5f*pi,\n'], min_omegas(1), min_omegas(2)')
		fprintf(['    instead of the specified ones at omega_s1 = %.5f*pi and ',...
	    'omega_s2 = %.5f*pi \n'], omega_s1, omega_s2')
	end
	fprintf(['(C) Minimized A_p = %.5e dB, ',...
	'instead of the specified A_p = %.5e dB\n'], AA_p, A_p)
	fprintf(['(D) Maximized A_s = %.5f dB, ',...
	'instead of the specified A_s = %.5f dB\n'], AA_s, A_s)
	disp(' ')
	disp(['I am ready to finalize one out of the above four filter ',...
	'designs. Hence, give'])
	Iopt=input('"1" for  (A), "2" for  (B), "3" for (C), and "4" for (D): ');
	%==========================================================================
	% Return to the design of the prototype filter of order N==N_low and
	% allocate the additional design margin, according to the above selection, 
	% to this filter.
	%==========================================================================
	ome_p=omega_p;
	ome_s=omega_s;
	AAA_p=A_p;
	AAA_s=A_s;
	if Iopt==1
	    ome_p=oomega_p;
	elseif Iopt==2
	    ome_s=oomega_s;
	elseif Iopt==3
	    AAA_p=AA_p;
	elseif Iopt==4
	    AAA_s=AA_s; 
	end
	%==========================================================================
	% Determine the lowpass prototype filter
	%==========================================================================
	N=N_low;
	[poles,zero_angles, scale]=...
	ClassicIIR_LP_poleszeros(N, AAA_p, ome_p, ome_s,iirtype);
	%==========================================================================
	if itype==1
	    %======================================================================
	    % Determine the zeros of the lowpass filter based on their zero angles;
	    % poles are included in poles and the scaling constant in scale
	    %=====================================================================
	    zeroes=exp(j*zero_angles);
	end
	if itype==2
	    %======================================================================
	    % The poles and zeros for the highpass filter are obtained by the 
	    % trivial substitution z^{-1} = - z^{-1}. The scaling constant is the
	    % same as for the lowpass prototype.
	    %======================================================================
	    % The following arrangements are used for ordering in the sequel the
	    % poles and zeros in an appropriate manner.
	    poles=-real(poles)+j*imag(poles);
	    zeroes=exp(j*(pi-zero_angles));
	end
	if itype==3
	    %======================================================================
	    % Start-up: Data for the prototype filter
	    %======================================================================
	    scale_LP=scale;
	    zeroes_LP=exp(j*zero_angles);
	    poles_LP=poles;
	    %======================================================================
	    % Poles of the bandpass filter
	    %======================================================================
	    poles=[];
	    for kk=1:floor((N+1)/2)
	        c(1) = gamma_2*poles_LP(kk)+1;
	        c(2) = -gamma_1*(poles_LP(kk)+1);
	        c(3) = poles_LP(kk)+gamma_2;
	        aa=roots(c).';
	        % Select two poles to be upper half plane complex poles or real 
	        % poles. If they are complex, allow the one with the smaller angle 
	        % to occur first. For real poles, select first the one with a more 
	        % positive radius. 
	        aa=real(aa)+j*abs(imag(aa));
	        if abs(imag(aa(1))) < 0.00001
	            aa=sort(aa,'descend');
			elseif angle(aa(2)) < angle(aa(1))
	            aa=[aa(2) aa(1)];
			end
	        poles=[poles aa];
		end
	    if rem(N,2)==0
	        % All poles occur in complex conjugates
	        % --->
	        poles=[poles flip(conj(poles))];
		else
	        if abs(imag(poles(2*floor((N+1)/2)-1)))> 0.00001 
	            % poles(2*floor((N+1)/2)-1) is complex, that is,
	            % poles(2*floor((N+1)/2)-1) and poles(2*floor((N+1)/2)) occur
	            % in complex conjugates
	            % --->
	            poles=[poles(1:end-1) flip(conj(poles(1:end-1)))];
			else
	            % poles(2*floor((N+1)/2)-1) is real, that is,
	            % poles(2*floor((N+1)/2)-1) and poles(2*floor((N+1)/2)) are 
	            % real
	            % --->
	            poles=[poles flip(conj(poles(1:end-2)))];
			end
		end
	    %======================================================================
	    % Zeros of the bandpass filter
	    %======================================================================
	    zeroes=[];
	    for kk=1:floor((N+1)/2)
	        c(1) = gamma_2*zeroes_LP(kk)+1;
	        c(2) = -gamma_1*(zeroes_LP(kk)+1);
	        c(3) = zeroes_LP(kk)+gamma_2;
	        aa=roots(c).';
	        % Follow what was mentioned above for the poles
	        aa=real(aa)+j*abs(imag(aa));
	        if abs(imag(aa(1))) < 0.00001
	            aa=sort(aa,'descend');
			elseif angle(aa(2)) < angle(aa(1))
	            aa=[aa(2) aa(1)];
			end
	        zeroes=[zeroes aa];
		end
	    if rem(N,2)==0
	        zeroes=[zeroes flip(conj(zeroes))];
		else
	        % For N odd, zeroes(N) and zeroes(N+1) are ALWAYS real
	        % --->
	        zeroes=[zeroes flip(conj(zeroes(1:end-2)))];
		end
	    %======================================================================
	    % Scaling constant 
	    %======================================================================
	    scale=scale_LP*prod(1+gamma_2*zeroes_LP)/prod(1+gamma_2*poles_LP);
	end
	if itype==4
	    %======================================================================
	    % Start-up: Data for the prototype filter
	    %======================================================================
	    scale_LP=scale;
	    zeroes_LP=exp(j*zero_angles);
	    poles_LP=poles;
	    %======================================================================
	    % Poles of the bandstop filter
	    %======================================================================
	    poles=[];
	    for kk=1:floor((N+1)/2)
	        c(1) = gamma_2*poles_LP(kk)-1;
	        c(2) = -gamma_1*(poles_LP(kk)-1);
	        c(3) = poles_LP(kk)-gamma_2;
	        aa=roots(c).';
	        % See the corresponding bandpass case
	        aa=real(aa)+j*abs(imag(aa));
	        if abs(imag(aa(1))) < 0.00001
	            aa=sort(aa,'descend');
			elseif angle(aa(2)) < angle(aa(1))
	            aa=[aa(2) aa(1)];
			end
	        poles=[poles aa];
		end
	    if rem(N,2)==0
	        % All poles occur in complex conjugate pairs
	        % --->
	        poles=[poles flip(conj(poles))];
		else
	        if abs(imag(poles(2*floor((N+1)/2)-1)))< 0.00001 
	            % poles(2*floor((N+1)/2)-1) is real, that is,
	            % poles(2*floor((N+1)/2)-1) and poles(2*floor((N+1)/2)) are 
	            % real 
	            % --->
	            poles=[poles flip(conj(poles(1:end-2)))];
			else
	            % poles(2*floor((N+1)/2)-1) is complex, that is,
	            % poles(2*floor((N+1)/2)-1) and poles(2*floor((N+1)/2)) occur
	            % in a complex conjugate pair 
	            % --->
	            poles=[poles(1:end-1) flip(conj(poles(1:end-1)))]; 
			end
		end
	    %======================================================================
	    % Zeros of the bandstop filter
	    %======================================================================
	    zeroes=[];
	    for kk=1:floor((N+1)/2)
	        c(1) = gamma_2*zeroes_LP(kk)-1;
	        c(2) = -gamma_1*(zeroes_LP(kk)-1);
	        c(3) = zeroes_LP(kk)-gamma_2;
	        aa=roots(c).';
	        % See the corresponding bandpass case
	        aa=real(aa)+j*abs(imag(aa));
	        if abs(imag(aa(1))) < 0.00001
	            aa=sort(aa,'descend');
			elseif angle(aa(2)) < angle(aa(1))
	            aa=[aa(2) aa(1)];
			end
	        zeroes=[zeroes aa];
		end
	    if rem(N,2)==0
	        zeroes=[zeroes flip(conj(zeroes))];
		else
	        % for N odd, zeroes(N) and zeroes(N+1) ALWAYS occur in complex 
	        % conjugates
	        % --->
	        zeroes=[zeroes(1:end-1) flip(conj(zeroes(1:end-1)))];
		end
	    %======================================================================
	    % Scaling constant
	    %======================================================================
	    scale=scale_LP*prod(1-gamma_2*zeroes_LP)/prod(1-gamma_2*poles_LP);    
	end
	%==========================================================================
	% Indices to implementing the filters as a parallel connection of two 
	% all-pass filters. These will be used while separating the poles between 
	% two all-pass filter sections.
	% For the prototype, the poles (zeros) have erlier been  ordered such that
	% (1) the upper half-plane complex poles (zeros) are arranged such that the
	%     pole (zero) with the largest radius (smallest angle) is the very 
	%     first pole (zero) and the remaining ones are ordered according to the 
	%     decreasing radii (increasing angle),
	% (2) the possible real pole (zero) for odd filter occurs after the upper  
	%     half-plane complex poles (zeros),
	% (3) Compared with their upper half-plane counterparts, the lower 
	%     half-plane complex poles (zeros) are arranged reversely with respect 
	%     to their angles.  
	% In the bandpass and bandstop cases, following the ordering of the 
	% prototype filter poles (zeros), for each upper half-plane complex pole 
	% (zero), two poles (zeros) have earlier been generated using, 
	% respectively, the substitutions 
	%
	%                 gamma_2-gamma_1z^{-1}+z^{-2}
	%      Z^{-1} = - ----------------------------
	%                 1-gamma_1z^{-1}+gamma_2z^{-2}
	%
	% and
	%
	%                 gamma_2-gamma_1z^{-1}+z^{-2}
	%      Z^{-1} = - ----------------------------- .
	%                 1-gamma_1z^{-1}+gamma_2z^{-2}
	%
	% The resulting two poles (zeros) are arranged according to the decreasing 
	% angle. Corresponding to a possible real pole (zero), in turn, there 
	% is a complex conjugate pair and two real-valued poles (zeros). In case of 
	% a complex conjugate, the upper half-plane one has first been used. 
	% Otherwise, two real-valued poles (zeros) have been arranged such that the 
	% one with the most positive real value occurs first. The lower half-plane 
	% poles (zeros) have treated as in the prototype filter case.
	% In the high pass case, the simple substitution Z^{-1} = z^{-1} is in use. 
	% In this case, obviously, what applies to the prototype is also valid for  
	% the resulting high pass filter.
	% In the prototype and highpass filter cases, the indexing for separating 
	% the two allpass filters is arranged so that
	% (1) every second pole belongs to the first (second) allpass filter,
	% (2) the first allpass filter contains the real pole.
	% The indexing for bandpass and bandstopb filters follows the above-
	% mentioned procedure used for generating their poles based on the 
	% prototype filter poles.
	%==========================================================================
	if Jopt==1
	    if itype==1 | itype==2
			% For itype=3 and itype=4, poles_LP has already been specified
	        poles_LP=poles;
		end
		ind1=1:2:length(poles_LP);
		ind2=2:2:length(poles_LP);
	    if rem(length(ind1),2)==1
	        % the first branch contains the real pole
	        ind1_LPHP=ind1;
	        ind2_LPHP=ind2;
		else
	        ind1_LPHP=ind2;
	        ind2_LPHP=ind1;
		end
	    if itype==3 | itype==4
	        ind1_BPBS=[2*ind1_LPHP 2*ind1_LPHP-1];
	        ind2_BPBS=[2*ind2_LPHP 2*ind2_LPHP-1];
	        ind1_BPBS=sort(ind1_BPBS);
	        ind2_BPBS=sort(ind2_BPBS);
		end
	end
	%==========================================================================
	% Time for the results
	%==========================================================================
	disp(' ')
	disp(' ')
	disp('TIME FOR THE RESULTS:')
	%==========================================================================
	% The "used" band edges for all filter types for plotting purposes
	%==========================================================================
	if Iopt~=1
	    if itype==1
	        use_omegap=omega_p;
		elseif itype==2
	        use_omegap=1-omega_p;
		else
	        use_omegap=real_omegap;
		end
	else
	    if itype==1
	        use_omegap=oomega_p;
		elseif itype==2
	        use_omegap=1-oomega_p;
		else
	        use_omegap=max_omegap;
		end
	end
	if Iopt~=2
	    if itype==1
	        use_omegas=omega_s;
		elseif itype==2
	        use_omegas=1-omega_s;
		else
	        use_omegas=real_omegas;
		end
	else
	    if itype==1
	        use_omegas=oomega_s;
		elseif itype==2
	        use_omegas=1-oomega_s;
		else
	        use_omegas=min_omegas;
		end
	end
	%==========================================================================
	% Print the poles, zeros, and scaling constant for the overall transfer 
	% function.
	%==========================================================================
	% While printing, simultaneously evaluate the magnitude, phase and group 
	% delay responses - why not because all the formulae are available in
	% http://www.cs.tut.fi/~ts/part2.pdf
	% Some modifications are, however, needed: 
	% (1) In case of the group delay expressions on page 57, add to the 
	%     formulas with the superscripts (zr), (zc), (pr), and (pc), +1, +2, 
	%     -1,and -2, respectively. Was it another way around - let me check!
	% (2) The formulae for the magnitude responses are OK.
	% (3) The formulae for the phase responses seems to valid according to the 
	%     present MATLAB unwrap command until the radius of the complex-
	%     conjugate zero does not exceed the value of unity. Then, something 
	%     weird happens either in THE FORMULA IN THE LECTURE NOTES or FREQZ! 
	%     For sure, the uwrap is required. The case with off-the-unit-circle 
	%     zeros is not needed now because all the zeros stay on the unit 
	%     circle.
	%==========================================================================
	% The lowpass and highpass filters and the bandpass and bandstop filters 
	% are considered separately due to the differences in their indices.
	%==========================================================================
	w=0:.000025:1;
	Magn_res=scale*ones(1,length(w));
	Phase_res=zeros(1,length(w));
	Group_res=zeros(1,length(w));
	Nove=length(poles);
	disp(' ')
	disp('The overall transfer function is')
	disp(' ')
	fprintf('           prod_{k=1}^{%d}[1-zeros(k)z^{-1}] \n', Nove)
	fprintf('H(z)=scale*--------------------------------- ,\n')
	fprintf('           prod_{k=1}^{%d}[1-poles(k)z^{-1}] \n', Nove)
	disp(' ')
	fprintf('where the %d poles are located at \n', Nove)
	if itype==3  itype==4
	    N1=rem(Nove/2,2);
	    N2=floor(Nove/2)-N1;
	    if N1==1
	        if abs(imag(poles(N1+N2)))< 1.0e-10
	            r1=real(poles(N2+N1));
	            r2=real(poles(N2+N1+1));
	            fprintf('z = %.17f\n', r1)
	            fprintf('z = %.17f,\n', r2)
	            Magn_res=Magn_res./sqrt(1-2*r1*cos(w*pi)+r1*r1);
	            Magn_res=Magn_res./sqrt(1-2*r2*cos(w*pi)+r2*r2);
	            Phase_res=Phase_res-atan2(r1*sin(w*pi),1-r1*cos(w*pi));
	            Phase_res=Phase_res-atan2(r2*sin(w*pi),1-r2*cos(w*pi));
	            Group_res=Group_res+...
				((1-r1*cos(w*pi))./(1-2*r1*cos(w*pi)+r1*r1)-1);
	            Group_res=Group_res+...
				((1-r2*cos(w*pi))./(1-2*r2*cos(w*pi)+r2*r2)-1); 
			else
	            r=abs(poles(N2+N1));
	            theta=angle(poles(N2+N1))/pi;
	            fprintf('z = %.17fexp(+-j%.17f*pi)\n',r, theta)
	            Magn_res=Magn_res./sqrt((1-2*r*cos((w-theta)*pi)+r*r).*...
				(1-2*r*cos((w+theta)*pi)+r*r));
	            Phase_res=Phase_res-...
				atan2(r*sin((w-theta)*pi),1-r*cos((w-theta)*pi))-...
				atan2(r*sin((w+theta)*pi),1-r*cos((w+theta)*pi));
	            Group_res=Group_res+...
				((1-r*cos((w-theta)*pi))./(1-2*r*cos((w-theta)*pi)+r*r)+...
				(1-r*cos((w+theta)*pi))./(1-2*r*cos((w+theta)*pi)+r*r)-2);
			end   
		end
	    for kk=1:N2-1
	        r=abs(poles(N2+1-kk));
	        theta=angle(poles(N2+1-kk))/pi;
	        fprintf('z = %.17fexp(+-j%.17f*pi)\n', r,theta)
	        Magn_res=Magn_res./sqrt((1-2*r*cos((w-theta)*pi)+r*r).*...
			(1-2*r*cos((w+theta)*pi)+r*r));
	        Phase_res=Phase_res-...
			atan2(r*sin((w-theta)*pi),1-r*cos((w-theta)*pi))-...
			atan2(r*sin((w+theta)*pi),1-r*cos((w+theta)*pi));
	        Group_res=Group_res+...
			((1-r*cos((w-theta)*pi))./(1-2*r*cos((w-theta)*pi)+r*r)+...
			(1-r*cos((w+theta)*pi))./(1-2*r*cos((w+theta)*pi)+r*r)-2);
		end
	    r=abs(poles(1));
	    theta=angle(poles(1))/pi;
	    fprintf('z = %.17fexp(+-j%.17f*pi),\n', r,theta)
	    Magn_res=Magn_res./sqrt((1-2*r*cos((w-theta)*pi)+r*r).*...
		(1-2*r*cos((w+theta)*pi)+r*r));
	    Phase_res=Phase_res-...
		atan2(r*sin((w-theta)*pi),1-r*cos((w-theta)*pi))-...
		atan2(r*sin((w+theta)*pi),1-r*cos((w+theta)*pi));
	    Group_res=Group_res+...
		((1-r*cos((w-theta)*pi))./(1-2*r*cos((w-theta)*pi)+r*r)+...
		(1-r*cos((w+theta)*pi))./(1-2*r*cos((w+theta)*pi)+r*r)-2);
	    fprintf('the %d zeroes at\n', Nove')
	    if N1==1
	        if abs(imag(zeroes(N2+N1)))< 1.0e-10
	            disp('z = 1')
	            disp('z = -1')
	            R1=1;
	            R2=-1;
	            Magn_res=Magn_res.*sqrt(1-2*R1*cos(w*pi)+R1*R1);
	            Magn_res=Magn_res.*sqrt(1-2*R2*cos(w*pi)+R2*R2);
	            Phase_res=Phase_res+atan2(R1*sin(w*pi),1-R1*cos(w*pi));
	            Phase_res=Phase_res+atan2(R2*sin(w*pi),1-R2*cos(w*pi));
	            Group_res=Group_res+...
				(-(1-R1*cos(w*pi))./(1-2*R1*cos(w*pi)+R1*R1)+1);
	            Group_res=Group_res+...
				(-(1-R2*cos(w*pi))./(1-2*R2*cos(w*pi)+R2*R2)+1);
			else
	            Theta=angle(zeroes(N2+N1))/pi;
	            fprintf('z = exp(+-j%.17f*pi)\n', Theta)
	            R=1;
	            Magn_res=Magn_res.*sqrt((1-2*R*cos((w-Theta)*pi)+R*R).*...
				(1-2*R*cos((w+Theta)*pi)+R*R));
	            Phase_res=Phase_res+...
				atan2(R*sin((w-Theta)*pi),1-R*cos((w-Theta)*pi))+...
				atan2(R*sin((w+Theta)*pi),1-R*cos((w+Theta)*pi));
	            Group_res=Group_res+...
				(-(1-R*cos((w-Theta)*pi))./(1-2*R*cos((w-Theta)*pi)+R*R)...
				-(1-R*cos((w+Theta)*pi))./(1-2*R*cos((w+Theta)*pi)+R*R)+2);
			end   
		end
	    for kk=1:N2-1
	        Theta=angle(zeroes(N2+1-kk))/pi;
	        fprintf('z = exp(+-j%.17f*pi)\n', Theta)
	        R=1;
	        Magn_res=Magn_res.*sqrt((1-2*R*cos((w-Theta)*pi)+R*R).*...
			(1-2*R*cos((w+Theta)*pi)+R*R));
	        Phase_res=Phase_res+...
			atan2(R*sin((w-Theta)*pi),1-R*cos((w-Theta)*pi))+...
			atan2(R*sin((w+Theta)*pi),1-R*cos((w+Theta)*pi));
	        Group_res=Group_res+...
			(-(1-R*cos((w-Theta)*pi))./(1-2*R*cos((w-Theta)*pi)+R*R)-...
			(1-R*cos((w+Theta)*pi))./(1-2*R*cos((w+Theta)*pi)+R*R)+2);
		end
	    Theta=angle(zeroes(1))/pi;
	    fprintf('z = exp(+-j%.17f*pi),\n', Theta)
	    R=1;
		Magn_res=Magn_res.*sqrt((1-2*R*cos((w-Theta)*pi)+R*R).*...
		(1-2*R*cos((w+Theta)*pi)+R*R));
	    Phase_res=Phase_res+...
		atan2(R*sin((w-Theta)*pi),1-R*cos((w-Theta)*pi))+...
		atan2(R*sin((w+Theta)*pi),1-R*cos((w+Theta)*pi));
	    Group_res=Group_res+...
		(-(1-R*cos((w-Theta)*pi))./(1-2*R*cos((w-Theta)*pi)+R*R)-...
		(1-R*cos((w+Theta)*pi))./(1-2*R*cos((w+Theta)*pi)+R*R)+2);
	    disp('and')
	    fprintf('scale = %.10f. \n', scale')
	else %itype=1 or itype=2
	    N1=rem(Nove,2);
	    N2=floor(Nove/2);
	    if N1==1
	        r=real(poles(N2+N1));
	        fprintf('z = %.11f\n', r)
	        Magn_res=Magn_res./sqrt(1-2*r*cos(w*pi)+r*r);
	        Phase_res=Phase_res-atan2(r*sin(w*pi),1-r*cos(w*pi));
	        Group_res=Group_res+...
			((1-r*cos(w*pi))./(1-2*r*cos(w*pi)+r*r)-1);
		end
	    for kk=1:N2-1
	        NN=N2-(kk-1);
	        r=abs(poles(NN));
	        theta=angle(poles(NN))/pi;
	        fprintf('z = %.17fexp(+-j%.17f*pi)\n', r,theta)
	        Magn_res=Magn_res./sqrt((1-2*r*cos((w-theta)*pi)+r*r).*...
			(1-2*r*cos((w+theta)*pi)+r*r));
	        Phase_res=Phase_res-...
			atan2(r*sin((w-theta)*pi),1-r*cos((w-theta)*pi))-...
			atan2(r*sin((w+theta)*pi),1-r*cos((w+theta)*pi));
	        Group_res=Group_res+...
			((1-r*cos((w-theta)*pi))./(1-2*r*cos((w-theta)*pi)+r*r)+...
			(1-r*cos((w+theta)*pi))./(1-2*r*cos((w+theta)*pi)+r*r)-2);
		end
	    r=abs(poles(1));
	    theta=angle(poles(1))/pi;
	    fprintf('z = %.17fexp(+-j%.17f*pi),\n', r,theta)
	    Magn_res=Magn_res./sqrt((1-2*r*cos((w-theta)*pi)+r*r).*...
		(1-2*r*cos((w+theta)*pi)+r*r));
	    Phase_res=Phase_res-...
		atan2(r*sin((w-theta)*pi),1-r*cos((w-theta)*pi))-...
		atan2(r*sin((w+theta)*pi),1-r*cos((w+theta)*pi));
	    Group_res=Group_res+...
		((1-r*cos((w-theta)*pi))./(1-2*r*cos((w-theta)*pi)+r*r)+...
		(1-r*cos((w+theta)*pi))./(1-2*r*cos((w+theta)*pi)+r*r)-2);
	    fprintf('the %d zeroes at\n', Nove')
	    if N1==1
	        if itype==1
	            disp('z = -1')
	            R=-1;
			else
				disp('z = 1')
				R=1; 
			end
	        Magn_res=Magn_res.*sqrt(1-2*R*cos(w*pi)+R*R);
	        Phase_res=Phase_res+atan2(R*sin(w*pi),1-R*cos(w*pi));
	        Group_res=Group_res+...
			(-(1-R*cos(w*pi))./(1-2*R*cos(w*pi)+R*R)+1);
		end
	    for kk=1:N2-1
	        NN=N2-(kk-1);
	        Theta=angle(zeroes(NN))/pi;
	        fprintf('z = exp(+-j%.17f*pi)\n', Theta)
	        R=1;
	        Magn_res=Magn_res.*sqrt((1-2*R*cos((w-Theta)*pi)+R*R).*...
			(1-2*R*cos((w+Theta)*pi)+R*R));
	        Phase_res=Phase_res+...
			atan2(R*sin((w-Theta)*pi),1-R*cos((w-Theta)*pi))+...
			atan2(R*sin((w+Theta)*pi),1-R*cos((w+Theta)*pi));
	        Group_res=Group_res+...
			(-(1-R*cos((w-Theta)*pi))./(1-2*R*cos((w-Theta)*pi)+R*R)...
			-(1-R*cos((w+Theta)*pi))./(1-2*R*cos((w+Theta)*pi)+R*R)+2);
		end
	    Theta=angle(zeroes(1))/pi;
	    fprintf('z = exp(+-j%.17f*pi),\n', Theta)
	    R=1;
		Magn_res=Magn_res.*sqrt((1-2*R*cos((w-Theta)*pi)+R*R).*...
		(1-2*R*cos((w+Theta)*pi)+R*R));
	    Phase_res=Phase_res+...
		atan2(R*sin((w-Theta)*pi),1-R*cos((w-Theta)*pi))+...
		atan2(R*sin((w+Theta)*pi),1-R*cos((w+Theta)*pi));
	    Group_res=Group_res+...
		(-(1-R*cos((w-Theta)*pi))./(1-2*R*cos((w-Theta)*pi)+R*R)...
		-(1-R*cos((w+Theta)*pi))./(1-2*R*cos((w+Theta)*pi)+R*R)+2);
	    disp('and')
	    fprintf('scale = %.10f. \n', scale')
	end
	%==========================================================================
	%                            FIGURE(1) 
	%                          Pole-Zero Plot
	%==========================================================================
	figure(1)
	zplane(zeroes.', poles.')
	title('Pole-Zero Plot for the Overall Filter') 
	%==========================================================================
	%                            FIGURE(2) 
	%                     Overall magnitude response
	%==========================================================================
	Ove_res=Magn_res;
	figure(2)
	plot(w,20*log10(abs(Ove_res)),'b','LineWidth',1.5); grid
	hold on
	if itype==1
	    plot([use_omegas 1], [-AAA_s -AAA_s],'r','LineWidth',1.5)
	elseif itype==2
	    plot([0 use_omegas], [-AAA_s -AAA_s],'r','LineWidth',1.5)
	elseif itype==3
	    plot([0 use_omegas(1)], [-AAA_s -AAA_s],'r','LineWidth',1.5)
	    hold on
	    plot([use_omegas(2) 1], [-AAA_s -AAA_s],'r','LineWidth',1.5)
	else
	    plot([use_omegas(1) use_omegas(2)], [-AAA_s -AAA_s]...
		,'r','LineWidth',1.5)
	end
	hold off
	ylabel('Magnitude Response in dB');
	xlabel('Angular Frequency \omega')
	title('Magnitude Response for the Overall Filter')
	pi_x_axis
	if itype==3 | itype==4
	    axis([0 1 -1.5*AAA_s .5*AAA_s])
	    if Iopt~=4
			addtext1 = ['A_s = ',num2str(AAA_s,'%.2f\n'), ' dB'];
		else
			addtext1 = ['Maximized A_s = ',num2str(AAA_s,'%.2f\n'), ' dB, '...
	        'instead of ',num2str(A_s,'%.2f\n'), ' dB'];
		end
	    text(.05,.4*AAA_s,addtext1,'fontsize',11);
	    if Iopt==2
	        addtext2 = ['"Minimized" \omega_{s1} = ',num2str(use_omegas(1)...
			,'%.5f\n'),'\pi and \omega_{s2} = ',...
			num2str(use_omegas(2),'%.5f\n'),'\pi, instead of'];
		else
	        addtext2 =['\omega_{s1} = ',num2str(use_omegas(1)...
			,'%.5f\n'),'\pi and \omega_{s2} = ',...
			num2str(use_omegas(2),'%.5f\n'),'\pi, instead of '];
		end
	    text(.05,.27*AAA_s,addtext2,'fontsize',11);
	    addtext3 = ['the specified at ',num2str(omega_s1...
		,'%.5f\n'),'\pi and ',num2str(omega_s2,'%.5f\n'),'\pi'];
	    text(.05,.14*AAA_s,addtext3,'fontsize',11);
	else
	    axis([0 1 -1.5*AAA_s .37*AAA_s])
	    if Iopt~=4
			addtext1 = ['A_s = ',num2str(AAA_s,'%.2f\n'), ' dB'];
		else
			addtext1 = ['Maximized A_s = ',num2str(AAA_s,'%.2f\n'), ' dB, '...
	        'instead of ',num2str(A_s,'%.2f\n'), ' dB'];
		end
	    text(.05,.27*AAA_s,addtext1,'fontsize',11);
	    if Iopt==2
	        if itype==1
	            addtext2 = ['Minimized \omega_s = ',num2str(use_omegas...
				,'%.5f\n'),'\pi, instead of ',num2str(omega_s...
				,'%.5f\n'),'\pi'];
			else
	            addtext2 = ['Maximized \omega_s = ',num2str(use_omegas...
				,'%.5f\n'),'\pi, instead of ',num2str(1-omega_s...
				,'%.5f\n'),'\pi'];
			end
		else
	        addtext2 = ['\omega_s = ',num2str(use_omegas,'%.5f\n'),'\pi' ];
		end
	    text(.05,.14*AAA_s,addtext2,'fontsize',11);
	end
	%==========================================================================
	%                            FIGURE(3) 
	%                        Passband details
	%==========================================================================
	figure(3)
	plot(w,20*log10(abs(Ove_res)),'b','LineWidth',1.5); grid
	hold on
	if itype==4
	    plot([0 use_omegap(1)], [-AAA_p -AAA_p],'r','LineWidth',1.5)
	    hold on
	    plot([0 use_omegap(1)], [0 0],'r','LineWidth',1.5)
	    hold on
	    plot([use_omegap(2) 1], [-AAA_p -AAA_p],'r','LineWidth',1.5)
	    hold on
	    plot([use_omegap(2) 1], [0 0],'r','LineWidth',1.5)
	    hold off
	else
	    if itype==3
	        plot([use_omegap(1) use_omegap(2)], [-AAA_p -AAA_p]...
			,'r','LineWidth',1.5)
	        hold on
	        low=use_omegap(1);
	        lower=use_omegap(1)-.1*(use_omegap(2)-use_omegap(1));
	        upper=use_omegap(2)+.1*(use_omegap(2)-use_omegap(1));
	        if Iopt==1
	            axis([lower upper  -1.3*AAA_p 0])% the exception
				%else
				%    axis([lower upper  -1.22*AAA_p 0])%upwards
			end        
		else
	        if itype==1
	            plot([0 use_omegap], [-AAA_p -AAA_p],'r','LineWidth',1.5)
	            hold on
	            low=0;
	            lower=0;
	            upper=0+1.2*(use_omegap);
			else
	            plot([use_omegap 1], [-AAA_p -AAA_p],'r','LineWidth',1.5)
	            hold on
	            low=use_omegap;
	            lower=use_omegap-.2*(1-use_omegap);
	            upper=1;
			end
		end
	end
	ylabel('Magnitude Response in dB');
	xlabel('Angular Frequency \omega')
	title('Passband Details for the Overall Filter')
	pi_x_axis
	if Iopt==3
	    if itype==4
	        addtext1  = ['Minimized A_p = ',num2str(AAA_p,'%.4e\n'), ' dB '...
			', instead of ',num2str(A_p,'%.4e\n'), ' dB'];
		else
	        addtext1  = ['Minimized A_p = ',num2str(AAA_p,'%.4e\n'), ' dB '...
			', instead of '];
	        addtext2  = [num2str(A_p,'%.4e\n'), ' dB'];
		end
	else
	    addtext1  = ['A_p = ',num2str(AAA_p,'%.4e\n'), ' dB'];
	end
	if itype==3 | itype==4
	    if Iopt==1
	        addtext3 = ['"Maximized" \omega_{p1} = ',num2str(use_omegap(1)...
			,'%.5f\n'),'\pi and \omega_{p2} = ',num2str(use_omegap(2)...
			,'%.5f\n'),'\pi,'];
	        addtext4 = ['instead of the specified at ',num2str(omega_p1...
			,'%.5f'),'\pi and ',num2str(omega_p2,'%.5f\n'),'\pi'];
		else
	        addtext3 = ['\omega_{p1} = ',num2str(use_omegap(1),'%.5f\n')...
			,'\pi and \omega_{p2} = ',num2str(use_omegap(2)...
			,'%.5f\n'),'\pi'];
		end
	else
	    if itype==1
	        if Iopt==1
	            addtext3 = ['Maximized \omega_p =',num2str(oomega_p...
				,'%.5f\n'),'\pi, instead  of ',...
				num2str(omega_p,'%.5f\n'),'\pi'];
			else
	            addtext3 = ['\omega_p = ',num2str(oomega_p,'%.5f\n'),'\pi'];
			end
		else
	        if Iopt==1
	            addtext3 = ['Minimized \omega_p = ',num2str(1-oomega_p...
				,'%.5f\n'),'\pi, instead  of ',...
				num2str(1-omega_p,'%.5f\n'),'\pi'];
			else
	            addtext3 = ['\omega_p = ',num2str(1-oomega_p,'%.5f\n'),'\pi'];
			end
		end
	end
	if itype==4 
	    if Iopt==1
	        axis([0 1  -1.1*AAA_p .32*AAA_p])
	        text(.05, 0.25*AAA_p,addtext1,'fontsize',11);
	        text(.05, 0.17*AAA_p,addtext3,'fontsize',11);
	        text(.05, 0.09*AAA_p,addtext4,'fontsize',11);
		else
	        axis([0 1  -1.1*AAA_p .24*AAA_p])
	        text(.05, 0.17*AAA_p,addtext1,'fontsize',11);
	        text(.05, 0.09*AAA_p,addtext3,'fontsize',11);
		end
	else
	    if itype==3 & Iopt==1
	        axis([lower upper  -1.3*AAA_p 0])
	        text(low+.02,-1.07*AAA_p,addtext1,'fontsize',11);
	        text(low+.02,-1.15*AAA_p,addtext3,'fontsize',11);
	        text(low+.02,-1.23*AAA_p,addtext4,'fontsize',11);
		elseif Iopt==3
	        axis([lower upper  -1.3*AAA_p 0])
	        text(low+.02,-1.07*AAA_p,addtext1,'fontsize',11);
	        text(low+.02,-1.15*AAA_p,addtext2,'fontsize',11);
	        text(low+.02,-1.23*AAA_p,addtext3,'fontsize',11);
		else
	        axis([lower upper  -1.22*AAA_p 0])
	        text(low+.02,-1.07*AAA_p,addtext1,'fontsize',11);
	        text(low+.02,-1.15*AAA_p,addtext3,'fontsize',11);
		end
	end
	%==========================================================================
	%                            FIGURE(4) 
	%      In the bandpass or bandstop cases, the relation between the 
	%      overall magnitude response and that of the prototype filter
	%==========================================================================
	if itype==3 | itype==4
	    if itype==3
	        w_Pro=-1:.000025:1;
		else
	        w_Pro=0:.000025:2;
		end
	    Proto_res=scale_LP*ones(1,length(w_Pro));
	    for kk=1:length(poles_LP)
	        Proto_res=Proto_res.*(1-zeroes_LP(kk)*exp(-j*w_Pro*pi))./...
			(1-poles_LP(kk)*exp(-j*w_Pro*pi));
		end
		%--------------------------------------------------------------------------
		% Relation between theta and omega
		%--------------------------------------------------------------------------
		if itype==3
			y=gamma_1*sin(pi*w)-gamma_2*sin(2*pi*w);
			x=1-gamma_1*cos(pi*w)+gamma_2*cos(2*pi*w);
			theta=(-pi+2*w*pi+2*atan2(y,x))/pi;
		else
			y=gamma_1*sin(pi*w)-gamma_2*sin(2*pi*w);
			x=1-gamma_1*cos(pi*w)+gamma_2*cos(2*pi*w);
			theta=(2*w*pi+2*atan2(y,x))/pi;
		end
		%--------------------------------------------------------------------------
		figure(4)
		%--------------------------------------------------------------------------
		%-----------
		subplot(221)
		%-----------
		plot(20*log10(abs(Proto_res)),w_Pro,'b','LineWidth',1.5);
		set(gca,'XDir','reverse')
		if itype==3
			axis([-1.5*AAA_s 15 -1 1]);
			set(gca,'Ytick',[-1 -.75  -.5 -.25 0 .25 .5 .75 1])
		else
			axis([-1.5*AAA_s 15 0 2]);
			set(gca,'Ytick',[0 .25 .5  .75  1  1.25 1.5 1.75 2])
		end
		pi_y_axis
		ylabel('Angular Frequency \theta')
		xlabel('|H_{LP}(e^{j\theta})| in dB');
		if itype==3
			title(['                                                   '...
			'Relation between the bandpass filter H(z) and its '...
			'prototype H_{LP}(Z)'])
		else
			title(['                                                   '...
			'Relation between the bandstop filter H(z) and its '...
			'prototype H_{LP}(Z)'])
		end
		hold on
		if itype==3
			yy=[-ome_s -ome_p ome_p ome_s];
		else
			yy=[ome_p ome_s 2-ome_s 2-ome_p];
		end
	
		magfix=[15 -1.5*AAA_s];
		if itype==3
			xx=[use_omegas(1) use_omegap(1) use_omegap(2) use_omegas(2)];
		else
			xx=[use_omegap(1) use_omegas(1) use_omegas(2) use_omegap(2)];
		end
		for nn=1:4
			plot(magfix,[yy(nn) yy(nn)],'black-.','LineWidth',0.8)
			hold on
		end
		hold off
		%-----------
		subplot(222)
		%-----------
		plot(w,theta,'b','LineWidth',1.5);
		%axis([0 1 -1 1]);
		xlabel('Angular Frequency \omega');
		ylabel('Angular Frequency \theta');
		if itype==3
			set(gca,'Ytick',[-1 -.75  -.5 -.25 0 .25 .5 .75 1])
			axis([0 1 -1 1]);
		else
			set(gca,'Ytick',[0 .25 .5  .75  1  1.25 1.5 1.75 2])
			axis([0 1 0 2]);
		end
		set(gca,'Xtick',[0 .2 .4 .6 .8 1])
		hold on
		pi_y_axis
		pi_x_axis
		for nn=1:4
			plot([0 xx(nn)],[yy(nn) yy(nn)],'black-.','LineWidth',0.8)
			hold on
			plot([xx(nn) xx(nn)],[-2 yy(nn)],'black-.','LineWidth',0.8)
		end
		hold off
		%-----------
		subplot(224)
		%-----------
		plot(w,20*log10(abs(Ove_res)),'b','LineWidth',1.5);
		hold on
		axis([0 1 -1.5*AAA_s 15]);
		set(gca,'Xtick',[0 .2 .4 .6 .8 1])
		%set(gca,'Ytick',[-80 -60 -40 -20 0])
		pi_x_axis
		for nn=1:4
			plot([xx(nn) xx(nn)],magfix,'black-.','LineWidth',0.8)
			hold on
		end
		xlabel('Angular Frequency \omega');
		ylabel('|H(e^{j\omega})| in dB');
		%-----------
		subplot(223)
		%-----------
		axis([0 10 0 10])
		axis off
		if itype==3
			addtext={['-\theta_p = ',num2str(-ome_p,'%.4f\n'),'\pi and '...
	        '\theta_p = ',num2str(ome_p,'%.4f\n'),'\pi'],'are mapped to',...
			['\omega_{p1} = ',num2str(use_omegap(1),'%.4f\n'),'\pi and '...
			'\omega_{p2} = ',num2str(use_omegap(2),'%.4f\n'),'\pi'],...
			['-\theta_s = ',num2str(-ome_s,'%.4f\n'),'\pi and \theta_s '...
			'= ',num2str(ome_s,'%.4f\n'),'\pi'],'are mapped to',...
			['\omega_{s1} = ',num2str(use_omegas(1),'%.4f\n'),'\pi and '...
			'\omega_{s2} = ',num2str(use_omegas(2),'%.4f\n'),'\pi']};
		else
			addtext={['\theta_p = ',num2str(ome_p,'%.4f\n'),'\pi and 2\pi-'...
	        '\theta_p = ',num2str(2-ome_p,'%.4f\n'),'\pi'],'are mapped to',...
			['\omega_{p1} = ',num2str(use_omegap(1),'%.4f\n'),'\pi and '...
			'\omega_{p2} = ',num2str(use_omegap(2),'%.4f\n'),'\pi'],...
			['\theta_s = ',num2str(ome_s,'%.4f\n'),'\pi and 2\pi-\theta_s '...
			'= ',num2str(2-ome_s,'%.4f\n'),'\pi'],'are mapped to',...
			['\omega_{s1} = ',num2str(use_omegas(1),'%.4f\n'),'\pi and '...
			'\omega_{s2} = ',num2str(use_omegas(2),'%.4f\n'),'\pi']};
		end
		text(-1 ,4.2,addtext)
	end
	%==========================================================================
	%                            FIGURE(5) 
	%                     Unwrapped Phase Response
	%==========================================================================
	Ove_phase=Phase_res;
	figure(5)
	plot(w,Ove_phase/pi,'b','LineWidth',1.5); grid
	min1=min(Ove_phase/pi);
	max1=max(Ove_phase/pi);
	ylabel('Phase Response');
	xlabel('Angular Frequency \omega')
	title('Overall Unwrapped Phase Response')
	pi_x_axis
	pi_y_axis
	axis([0 1 min1-.05*(max1-min1)  max1+.3*(max1-min1)])
	addtext={'Each time the filter has a zero pair at z=exp(+-j\theta),',...
	'there is a jump of \pi upwards at \omega = \theta.',...
	'The jumps upwards and downwards should alternate!'};
	text(.05,max1+.15*(max1-min1),addtext,'fontsize',12);
	%==========================================================================
	%                            FIGURE(6) 
	%                     Modified Phase Response
	%==========================================================================
	Mod_phase=Ove_phase;
	Nove=length(zeroes);
	if itype==4
	    ang_ove=sort(angle(zeroes(1:Nove/2)))/pi;
	elseif itype==3
	    N1=rem(Nove/2,2);
	    N2=floor(Nove/2)-N1;    
	    ang_ove=sort(angle(zeroes(1:N2)))/pi;
	else
	    ang_ove=sort(angle(zeroes(1:floor(Nove/2))))/pi;
	end
	for kk=2:2:length(ang_ove)
	    ind = find(w <= ang_ove(kk), 1, 'last' );
	    Mod_phase(ind+1:end)=Mod_phase(ind+1:end)-2*pi;
	end
	figure(6)
	plot(w,Mod_phase/pi,'b','LineWidth',1.5); grid
	ylabel('Phase Response');
	xlabel('Angular Frequency \omega')
	title('Modified Phase Response ')
	pi_x_axis
	pi_y_axis
	%==========================================================================
	%                            FIGURE(7) 
	%                     Passband Phase Delay
	%==========================================================================
	if itype==1
	    ind = find(w <= 1.2*use_omegap, 1, 'last' ); 
	    ww=w(2:ind);
	    Mod_phase_pass=Mod_phase(2:ind);
	    %Group_res_pass=Group_res(2:ind);
	elseif itype==2
	    ind=find(w >= use_omegap-0.2*(1-use_omegap), 1, 'first' );
	    ww=w(ind:end);
	    Mod_phase_pass=Mod_phase(ind:end);
	    %Group_res_pass=Group_res(ind:end);
	elseif itype==3
	    over_pass=use_omegap(2)-use_omegap(1);
	    ind1=find(w >= use_omegap(1)-0.1*over_pass, 1 ,'first');
	    ind2=find(w <= use_omegap(2)+0.1*over_pass, 1 ,'last');
	    ww=w(ind1:ind2);
	    Mod_phase_pass=Mod_phase(ind1:ind2);
	    %Group_res_pass=Group_res(ind1:ind2);
	else 
	    ind1=find(w <= 1.05*use_omegap(1), 1, 'last' ); 
	    ind2=find(w >= .95*use_omegap(2), 1, 'first' ); 
	    ww1=w(2:ind1);
	    Mod_phase_pass1=Mod_phase(2:ind1);
	    %Group_res_pass1=Group_res(2:ind1);
	    ww2=w(ind2:end);
	    Mod_phase_pass2=Mod_phase(ind2:end);
	    %Group_res_pass2=Group_res(ind2:end);
	end
	figure(7)
	if itype==4
	    plot(ww1,-Mod_phase_pass1./ww1,'b',ww2,...
		-Mod_phase_pass2./ww2,'b','LineWidth',1.5);grid
	else
	    plot(ww,-Mod_phase_pass./ww,'b','LineWidth',1.5);grid
	end
	hold on
	if itype~=4
	    min1=min(-Mod_phase_pass./ww);
	    max1=max(-Mod_phase_pass./ww);
	else
	    min1=min(min(-Mod_phase_pass1./ww1),min(-Mod_phase_pass2./ww2));
	    max1=max(max(-Mod_phase_pass1./ww1),max(-Mod_phase_pass2./ww2));
	end

	min2=min1-.05*(max1-min1);
	max2=max1+.25*(max1-min1);
	if itype==4
	    axis([ww1(1) ww2(end) min2 max2])
	else
	    axis([ww(1) ww(end) min2 max2])
	end
	if itype==1 | itype==2
		plot([use_omegap use_omegap],[min2 max2],'r-.','LineWidth',1.2)
		hold off
		addtext = ['\omega_p = ',num2str(use_omegap,'%.8f\n'),'\pi'];
		if itype==2
			text(use_omegap+.1*(1-use_omegap),...
			max1+.125*(max1-min1),addtext,'fontsize',11);
		else
			text(.25*use_omegap,max1-.125*(max1-min1),addtext,'fontsize',11);    
		end
	else
	    plot([use_omegap(1) use_omegap(1)],[min2 max2],'r-.','LineWidth',1.2)
	    hold on
	    plot([use_omegap(2) use_omegap(2)],[min2 max2],'r-.','LineWidth',1.2)
	    hold off   
	    addtext = {['\omega_{p1} = ',num2str(use_omegap(1),'%.8f\n'),'\pi'],...
		['\omega_{p2} = ',num2str(use_omegap(2),'%.8f\n'),'\pi']};
	    if itype==3
	        text(use_omegap(1)+.075*(use_omegap(2)-use_omegap(1)),...
			max2-.125*(max2-min2),addtext,'fontsize',11);
		else
	        text(use_omegap(2)+.05,...
			max1+.125*(max1-min1),addtext,'fontsize',11);
		end  
	end   
	ylabel('Phase Delay in Samples');
	xlabel('Angular Frequency \omega')
	title('Passband Phase Delay')
	pi_x_axis
	%==========================================================================
	%                            FIGURE(8) 
	%                       Group Delay Response
	%==========================================================================
	figure(8)
	plot(w,Group_res,'b','LineWidth',1.5); grid
	ylabel('Group Delay in Samples');
	xlabel('Angular Frequency \omega')
	title('Overall Group Delay')
	pi_x_axis
	%==========================================================================
	%                            FIGURE(9) 
	%                     Passband Group Delay
	%==========================================================================
	figure(9)
	%if itype==4
	%    plot(ww1,Group_res_pass1,'b',ww2,...
	%        Group_res_pass2,'b','LineWidth',1.5);grid
	%else
	%    plot(ww,Group_res_pass,'b','LineWidth',1.5);grid
	%end
	plot(w,Group_res,'b','LineWidth',1.5);grid
	hold on
	%if itype~=4
	%    min1=min(Group_res_pass);
	%    max1=max(Group_res_pass);
	%else
	%    min1=min(min(Group_res_pass1),min(Group_res_pass2));
	%    max1=max(max(Group_res_pass1),max(Group_res_pass2));
	%end
	min1=min(Group_res);  
	max1=max(Group_res);
	min2=min1-.05*(max1-min1);
	max2=max1+.25*(max1-min1);
	if itype==4
	    axis([ww1(1) ww2(end) min2 max2])
	else
	    axis([ww(1) ww(end) min2 max2])
	end
	if itype==1 | itype==2
		plot([use_omegap use_omegap],[min2 max2],'r-.','LineWidth',1.2)
		hold off
		addtext = ['\omega_p = ',num2str(use_omegap,'%.8f\n'),'\pi'];
		if itype==2
			text(use_omegap+.1*(1-use_omegap),...
			max1+.125*(max1-min1),addtext,'fontsize',11);
		else
			text(.25*use_omegap,max1-.125*(max1-min1),addtext,'fontsize',11);    
		end
	else
	    plot([use_omegap(1) use_omegap(1)],[min2 max2],'r-.','LineWidth',1.2)
	    hold on
	    plot([use_omegap(2) use_omegap(2)],[min2 max2],'r-.','LineWidth',1.2)
	    hold off   
	    addtext = {['\omega_{p1} = ',num2str(use_omegap(1),'%.8f\n'),'\pi'],...
		['\omega_{p2} = ',num2str(use_omegap(2),'%.8f\n'),'\pi']};
	    if itype==3
			text(use_omegap(1)+.075*(use_omegap(2)-use_omegap(1)),...
			max2-.125*(max2-min2),addtext,'fontsize',11);
		else
	        text(use_omegap(2)+.05,...
			max1+.125*(max1-min1),addtext,'fontsize',11);
		end  
	end   
	ylabel('Group Delay in Samples');
	xlabel('Angular Frequency \omega')
	title('Passband Group Delay')
	pi_x_axis
	%==========================================================================
	% Implementation of the filter as a as a parallel connection of two 
	% all-pass filters for Jopt==1. The lowpass and highpass filters and the 
	% bandpass and banstop filters are considered separately due to the 
	% differences in the indices.
	%==========================================================================
	if Jopt==1
	    if itype==1 | itype==2
	        ind1=ind1_LPHP;
	        ind2=ind2_LPHP;
		else
	        ind1=ind1_BPBS;
	        ind2=ind2_BPBS;
		end
	    N_A=length(ind1);
	    N_B=length(ind2);
	    disp(' ')
	    disp(['You were willing to implement the filter as a parallel ' ,...
		'connection of two all-pass filters. For'])
	    disp(['this purpose, the overall filter is implementable in the ',...
		'form H(z)=(1/2)[A(z)+B(z)], where'])
	    fprintf(['A(z) is the %dth-order all-pass filter whose denominator',...
		' is factorizable into terms \n'], N_A)
	    if itype==1 | itype==2
	        aa=real(poles(ind1((N_A+1)/2)));
	        if aa>=0
	            fprintf('1 - %.17f*z^{-1}\n', aa)
			else
	            fprintf('1 + %.17f*z^{-1}\n', -aa)
			end   
	        ang_A=-w*pi-2*atan2(aa*sin(pi*w),1-aa*cos(pi*w));
	        for nn = 1:(N_A-1)/2
	            aa=poles(ind1((N_A+1)/2-nn));
	            coef1=-2*abs(aa)*cos(angle(aa));
	            coef2=abs(aa)^2;
	            ang_A=ang_A-2*w*pi-2*atan2(-coef1*sin(pi*w)-...
				coef2*sin(2*pi*w),1+coef1*cos(pi*w)+coef2*cos(2*pi*w));
	            if coef1>=0
	                fprintf('1 + %.17f*z^{-1} + %.17f*z^{-2}\n', coef1, coef2)
				else
	                fprintf('1 - %.17f*z^{-1} + %.17f*z^{-2}\n', -coef1, coef2)
				end  
			end
	        fprintf(['and B(z) is the %dth-order allpass filter whose ',...
			'denominator is factorizable into terms\n'], N_B)
	        ang_B=zeros(1,length(w));
	        for nn = 1:N_B/2
	            aa=poles(ind2(N_B/2-nn+1)); 
	            coef1=-2*abs(aa)*cos(angle(aa));
	            coef2=abs(aa)^2;
	            ang_B=ang_B-2*w*pi-2*atan2(-coef1*sin(pi*w)-...
				coef2*sin(2*pi*w),1+coef1*cos(pi*w)+coef2*cos(2*pi*w));
	            if coef1>=0
	                fprintf('1 + %.17f*z^{-1} + %.17f*z^{-2}\n', coef1, coef2)
				else
	                fprintf('1 - %.17f*z^{-1} + %.17f*z^{-2}\n', -coef1, coef2)
				end
			end
		else % That is, itype=3 or itype=4
	        NN=length(poles(ind1))/2;
	        ang_A=zeros(1,length(w));
	        for kk=1:NN-1
	            aa=poles(ind1(kk));
	            coef1=-2*abs(aa)*cos(angle(aa));
	            coef2=abs(aa)^2;
	            ang_A=ang_A-2*w*pi-2*atan2(-coef1*sin(pi*w)-...
				coef2*sin(2*pi*w),1+coef1*cos(pi*w)+coef2*cos(2*pi*w));
	            if coef1>=0
	                fprintf('1 + %.17f*z^{-1} + %.17f*z^{-2}\n', coef1, coef2)
				else
	                fprintf('1 - %.17f*z^{-1} + %.17f*z^{-2}\n', -coef1, coef2)
				end
			end
	        % two real poles or one complex conjugate pole pair - study
	        if imag(poles(ind1(NN))) < 1.0e-10
	            if real(poles(ind1(NN)))>0
	                fprintf('1 - %.17f*z^{-1}\n', real(poles(ind1(NN))))
				else
	                fprintf('1 + %.17f*z^{-1}\n', -real(poles(ind1(NN))))
				end
	            if real(poles(ind1(NN+1)))>0
	                fprintf('1 - %.17f*z^{-1}\n', real(poles(ind1(NN+1))))
				else
	                fprintf('1 + %.17f*z^{-1}\n', -real(poles(ind1(NN+1))))
				end
	            aa=real(poles(ind1(NN)));
	            bb=real(poles(ind1(NN+1)));
	            ang_A=ang_A-w*pi-2*atan2(aa*sin(pi*w),1-aa*cos(pi*w));
	            ang_A=ang_A-w*pi-2*atan2(bb*sin(pi*w),1-bb*cos(pi*w));
			else
	            aa=poles(ind1(NN));
	            coef1=-2*abs(aa)*cos(angle(aa));
	            coef2=abs(aa)^2;
	            ang_A=ang_A-2*w*pi-2*atan2(-coef1*sin(pi*w)-...
				coef2*sin(2*pi*w),1+coef1*cos(pi*w)+coef2*cos(2*pi*w));
	            if coef1>=0
	                fprintf('1 + %.17f*z^{-1} + %.17f*z^{-2}\n', coef1, coef2)
				else
	                fprintf('1 - %.17f*z^{-1} + %.17f*z^{-2}\n', -coef1, coef2)
				end
			end
	        fprintf(['and B(z) is the %dth-order allpass filter whose ',...
			'denominator is factorizable into terms\n'], N_B)
	        ang_B=zeros(1,length(w));
	        NN=length(poles(ind2))/2;
	        for kk=1:NN
	            aa=poles(ind2(kk));
	            bb=poles(ind2(NN+1-kk));
	            coef1=-2*abs(aa)*cos(angle(aa));
	            coef2=abs(aa)^2;
	            ang_B=ang_B-2*w*pi-2*atan2(-coef1*sin(pi*w)-...
				coef2*sin(2*pi*w),1+coef1*cos(pi*w)+coef2*cos(2*pi*w));
	            if coef1>=0
	                fprintf('1 + %.17f*z^{-1} + %.17f*z^{-2}\n', coef1, coef2)
				else
	                fprintf('1 - %.17f*z^{-1} + %.17f*z^{-2}\n', -coef1, coef2)
				end  
			end
		end
		%==========================================================================
		%                            FIGURE(10) 
		%              Phase Responses for the Two Allpass Filters
		%==========================================================================
	    figure(10)
	    plot(w,ang_A/pi,'r','LineWidth',1.5)
	    hold on
	    plot(w,ang_B/pi,'b','LineWidth',1.5);
	    hold off
	    grid
	    ylabel('Phase Responses');
	    xlabel('Angular Frequency \omega')
	    title(['Phase Responses for{\color{red} A(z)} and'...
		'{\color{blue} B(z)}'])
	    pi_x_axis
	    pi_y_axis
		%==========================================================================
		%                            FIGURE(11) 
		%              Magnitudes for the Desired Filter and its
		%                   Power-Complementary Counterpart
		%==========================================================================
	    HH=abs(.5*(exp(j*ang_A)+exp(j*ang_B)));
	    GG=abs(.5*(exp(j*ang_A)-exp(j*ang_B)));
	    figure(11)
	    if itype==1 | itype==4
	        plot(w,20*log10(abs(HH)),'r','LineWidth',1.5); grid
	        hold on
	        plot(w,20*log10(abs(GG)),'b','LineWidth',1.5);
		else
	        plot(w,20*log10(abs(HH)),'b','LineWidth',1.5); grid
	        hold on
	        plot(w,20*log10(abs(GG)),'r','LineWidth',1.5);
		end
	    axis([0 1 -1.5*AAA_s 0])
	    ylabel('Magnitude Responses in dB');
	    xlabel('Angular Frequency \omega')
	    if itype==1 | itype==4
	        title({['Responses for {\color{red}Desired [A(z)+B(z)]/2} '...
			'and its']; ['{\color {blue}Power-Complementary Counterpart '...
			'[A(z)-B(z)]/2}']})
		else
			title({['Responses for {\color{red}Desired [A(z)-B(z)]/2} '...
			'and its']; ['{\color {blue}Power-Complementary Counterpart '...
			'[A(z)+B(z)]/2}']}) 
		end
	    pi_x_axis
		end
